unit ordersDB;

interface

uses
  helper,
	System.Configuration,
	System.Data,
	Borland.Data.Provider,
  Borland.Data.Common;

//*******************************************************
//
// TOrderDetails Class
//
// A simple data class that encapsulates details about
// a particular order inside the DelphiWebStore Orders
// database.
//
//'*******************************************************
	type TOrderDetailsDB = class
		public
			OrderDate : TDateTime;
			ShipDate : TDateTime;
			OrderTotal : Decimal;
			OrderItems : System.Data.DataSet;
	end;
	
	
//*******************************************************
//
// TOrderHistoryDB Class
//
// Business/Data Logic Class that encapsulates all data
// logic necessary to query past orders within the
// DelphiWebStore Orders database.
//
//*******************************************************
	type TOrdersDB = class
		public
			function GetCustomerOrders(customerID : String) : BDPDataReader;
			function GetOrderDetails(orderID : Integer; customerID : String) : TOrderDetailsDB;
			function CalculateShippingDate(customerID, cartID : String) : System.DateTime;
			function PlaceOrder(customerID, cartID : String) : Integer;
	end;

implementation	

//*******************************************************
//
// TOrdersDB.GetCustomerOrders() Method <a name="GetCustomerOrders"></a>
//
// The GetCustomerOrders method returns a struct containing
// a forward-only, read-only DataReader.  This displays a list of all
// past orders placed by a specified customer.
// The SQLDataReaderResult struct also returns the SQL connection,
// which must be explicitly closed after the data from the DataReader
// is bound into the controls.
//
//*******************************************************
function TOrdersDB.GetCustomerOrders(customerID : String) : BDPDataReader;
var
	myConnection : BDPConnection;
	myCommand : BDPCommand;
	parameterCustomerID : BDPParameter;
begin
	// Create Instance of Connection and Command Object
	myConnection := BDPConnection.Create(THelper.ConnectionString);
  myConnection.ConnectionOptions := THelper.ConnectionOptions;
	myCommand := BDPCommand.create('SELECT * FROM OrdersList(?)', myConnection);

	// Mark the Command as a SPROC
	myCommand.CommandType := CommandType.Text;

	// Add Parameters to SPROC
	parameterCustomerID := BDPParameter.create('CustomerID', BDPType.Int32, 4);
	parameterCustomerID.Value := TObject(customerID);
	myCommand.Parameters.Add(parameterCustomerID);

	// Execute the command
	myConnection.Open;
	Result := myCommand.ExecuteReader(CommandBehavior.CloseConnection);
end;

//*******************************************************
//
// TOrdersDB.GetOrderDetails() Method <a name="GetOrderDetails"></a>
//
// The GetOrderDetails method returns an OrderDetails
// struct containing information about the specified
// order.
//
//*******************************************************
function TOrdersDB.GetOrderDetails(orderID : Integer; customerID : String) : TOrderDetailsDB;
var
	myConnection : BDPConnection;
	myCommand : BDPCommand;
  myDataSet : DataSet;
  myAdapter : BDPDataAdapter;
	parameterOrderID,
	parameterCustomerID,
	parameterOrderDate,
	parameterShipDate,
	parameterOrderTotal : BDPParameter;
	myOrderDetails : TOrderDetailsDB;
begin
	// Create Instance of Connection and Command Object
	myConnection := BDPConnection.Create(THelper.ConnectionString);
  myConnection.ConnectionOptions := THelper.ConnectionOptions;
	myCommand := BDPCommand.Create('ORDERSDETAIL_PARAMS', myConnection);

	// Mark the Command as a SPROC
	myCommand.CommandType := CommandType.StoredProcedure;

	// Add Parameters to SPROC
	parameterOrderID := BDPParameter.create('OrderID', BDPType.Int32, 4);
	parameterOrderID.Value := TObject(orderID);
	myCommand.Parameters.Add(parameterOrderID);

	parameterCustomerID := BDPParameter.Create('CustomerID', BDPType.Int32, 4);
	parameterCustomerID.Value := TObject(customerID);
	myCommand.Parameters.Add(parameterCustomerID);

	parameterOrderDate := BDPParameter.Create('OrderDate', BDPType.DateTime, 8);
	parameterOrderDate.Direction := ParameterDirection.Output;
	myCommand.Parameters.Add(parameterOrderDate);

	parameterShipDate := BDPParameter.Create('ShipDate', BDPType.DateTime, 8);
	parameterShipDate.Direction := ParameterDirection.Output;
	myCommand.Parameters.Add(parameterShipDate);

	parameterOrderTotal := BDPParameter.Create('OrderTotal', BDPType.Float, 8);
  parameterOrderTotal.Precision := 15;
  parameterOrderTotal.Scale := 4;
	parameterOrderTotal.Direction := ParameterDirection.Output;
	myCommand.Parameters.Add(parameterOrderTotal);


  myConnection.Open;

  myCommand.ExecuteNonQuery;
  myCommand.Parameters.Clear;

	// Create and Fill the DataSet
  myCommand.CommandText := 'SELECT * FROM ORDERSDETAIL_DATASET(?)';
  myCommand.CommandType := CommandType.Text;

	parameterOrderID := BDPParameter.create('OrderID', BDPType.Int32, 4);
	parameterOrderID.Value := TObject(orderID);
	myCommand.Parameters.Add(parameterOrderID);

  myDataSet := DataSet.Create;
  myAdapter := BDPDataAdapter.Create(myCommand);
	myAdapter.Fill(myDataSet, 'OrderItems');

  myConnection.Close;

	// ship date is null if order doesn't exist, or belongs to a different user
	if Assigned(parameterShipDate.Value) Then
	begin
		// Create and Populate OrderDetails Struct using
		// Output Params from the SPROC, as well as the
		// populated dataset from the BDPDataAdapter
		myOrderDetails := TOrderDetailsDB.Create;

		myOrderDetails.OrderDate := Convert.ToDateTime(parameterOrderDate.Value.ToString);
		myOrderDetails.ShipDate := Convert.ToDateTime(parameterShipDate.Value.ToString);
		myOrderDetails.OrderTotal := Convert.ToDecimal(parameterOrderTotal.Value);
		myOrderDetails.OrderItems := myDataSet;

    // Return the DataSet
    Result := myOrderDetails;
	end
 else
	begin
	 Result:= nil;
	end;
end;//TOrdersDB.GetOrderDetails

//*******************************************************
//
// TOrdersDB.CalculateShippingDate() Method <a name="CalculateShippingDate"></a>
//
// The CalculateShippingDate method would be where you would
// place all of the code necessary to calculate the shipping
// ETA.  For now, we are just making up a random date.
//
//*******************************************************
function TOrdersDB.CalculateShippingDate(customerID, cartID : String) : System.DateTime;
begin
 Result := DateTime.Now.AddMonths(1);
end;

//*******************************************************
//
// TOrdersDB.PlaceOrder() Method <a name="PlaceOrder"></a>
//
// The PlaceOrder method places an order within the
// DelphiWebStore Orders Database and then clears out the current
// items within the shopping cart.
//
//*******************************************************
function TOrdersDB.PlaceOrder(customerID, cartID : String) : Integer;
var
	myConnection : BDPConnection;
	myCommand : BDPCommand;
	parameterCustomerID,
	parameterCartID,
	parameterShipDate,
	parameterOrderDate,
	parameterOrderID : BDPParameter;
begin
	// Create Instance of Connection and Command Object
	myConnection := BDPConnection.Create(THelper.ConnectionString);
  myConnection.ConnectionOptions := THelper.ConnectionOptions;
	myCommand := BDPCommand.create('OrdersAdd', myConnection);

	// Mark the Command as a SPROC
	myCommand.CommandType := CommandType.StoredProcedure;

	// Add Parameters to SPROC
	parameterCustomerID := BDPParameter.Create('CustomerID', BDPType.Int32, 4);
	parameterCustomerID.Value := TObject(customerID);
	myCommand.Parameters.Add(parameterCustomerID);

	parameterCartID := BDPParameter.Create('CartID', BDPType.String, 50);
	parameterCartID.Value := cartID;
	myCommand.Parameters.Add(parameterCartID);

	parameterOrderDate := BDPParameter.Create('OrderDate', BDPType.DateTime, 8);
	parameterOrderDate.Value := DateTime.Now;
	myCommand.Parameters.Add(parameterOrderDate);

	parameterShipDate := BDPParameter.Create('ShipDate', BDPType.DateTime, 8);
	parameterShipDate.Value := TObject(CalculateShippingDate(customerID, cartID));
	myCommand.Parameters.Add(parameterShipDate);

	parameterOrderID := BDPParameter.Create('OrderID', BDPType.Int32, 4);
	parameterOrderID.Direction := ParameterDirection.Output;
	myCommand.Parameters.Add(parameterOrderID);

	// Open the connection and execute the Command
	myConnection.Open;
	myCommand.ExecuteNonQuery;
	myConnection.Close;

	// Return the OrderID
	Result := Integer(parameterOrderID.Value);
end;	//TOrdersDB.PlaceOrder

end.
